function [X,Y,iter,out] = lrtc(M,Data,Omega,L,opts)
% Solve the Low-rank Tensor Completion Problem
%
% [X,Y,iter,out] = lrtc(M,Data,L,gamma,opts)
%
% with:
% M:     the underlying tensor (n1 x n2 x n3)
% Data:  the observed tensor (n1 x n2 x n3)
% L:     orthogonal matrix (n3 x n3)
% gamma: index of the unobserved entries
% opts:  structure value in Matlab
%
% Written by

maxiter      = opts.maxiter;
lambdamax    = opts.lambdamax;
lambdamin    = opts.lambdamin;
mu           = opts.mu;
rho          = opts.rho;
tau          = opts.tau;
delta        = opts.delta;
tol          = opts.tol;

out          = [];
out.err      = zeros(maxiter,1);

lambda       = lambdamax;
[n1,n2,n3]   = size(Data);
k            = min(n1,n2);
X            = Data;
Y            = Data;
L_inv        = inv(L);


for iter = 1 : maxiter
    Xk = X;
    Yk = Y;
    
    % update Y
    
    temp  = mu*X + (1-mu)*Y;
    Y = prox_rank(temp,2*mu*lambda,L);
    
    % update X
    
    Temp = Y.*(1-Omega);
    X    = Data + Temp;
    
    
    errx  = max(abs(X(:)-Xk(:)));
    erry  = max(abs(Y(:)-Yk(:)));
    errxy = max(errx,erry);
    
    err                = norm(M(:)-X(:))/norm(M(:));
    out.err(iter)      = err;
    
    if iter == 1 || mod(iter,tau) == 0
        disp(['iter' num2str(iter) ',  lambda=' num2str(lambda) ...
            ', err=' num2str(err)]);
    end
    
    if errxy < delta
        lambda = max(lambda*rho, lambdamin);
    end
    
    if errxy~=0 && errxy < tol
        break;
    end
    
end

